///////////////////////////////////////////////////////////////////////////////
// (C) Copyright 2001 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// CREATED BY:
//	Valentin Ofshteyn, June 2001
//
// DESCRIPTION:
// AttrPosEnabler.cpp: this file includes the implementation 
//	of the AttrPosEnabler, MyExportReactor and MyConnectionReactor classes
//
///////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "AttrPosEnabler.h"
#include "dbents.h"
///////////////////////////////////////////////////////////////////////////////
//								ENABLER										 //
///////////////////////////////////////////////////////////////////////////////

//Declarations
//

//this array of strings defines the names of the columns in the table, 
//which contains the coordinates of the attributes.
const char*	AttrPosEnabler::sm_kpcParamNames[4]={
		"ORAID",
		"XPOS",
		"YPOS",
		"ZPOS"
};
//this array contains the column types for the table
const int AttrPosEnabler::sm_nParamTypes[4]={
		OTYPE_NUMBER,
		OTYPE_FLOAT,
		OTYPE_FLOAT,
		OTYPE_FLOAT
};

//Constructor
//
AttrPosEnabler::AttrPosEnabler(AcMapOSEConnection *pConnection):
	m_pConnection(pConnection),
	m_pImportReactor(NULL),
	m_pExportReactor(NULL),
	m_pConnectionReactor(NULL)
{
	if (m_pConnection)
	{
		//instantiate objects for notification checking (during 
		//import, export and connection)
		m_pImportReactor = new MyImportReactor(this);
		m_pExportReactor = new MyExportReactor(this);
		m_pConnectionReactor = new MyConnectionReactor(this);
		
		//attach the reators to the active connection
		m_pConnection->AddConnectionReactor(m_pConnectionReactor);		
		m_pConnection->AddImportReactor(m_pImportReactor);		
		m_pConnection->AddExportReactor(m_pExportReactor);		

		//create the table, which will contain the attribute positions
		if (m_pConnection->IsConnected())
		{
			m_clollParams=m_pConnection->Database().GetParameters();
			assert(m_clollParams.IsOpen());
			const char *kpcSQLCreateTable=
				"create table POSITIONS ("
				"EntityID NUMBER,"
				"XPOS NUMBER,"
				"YPOS NUMBER,"
				"ZPOS NUMBER)";
			oresult res=m_pConnection->Database().ExecuteSQL(kpcSQLCreateTable);
			//we can ignore the result of this operation if the table is already
			//created
		}
	}
}

AttrPosEnabler::~AttrPosEnabler()
{
	//remove the reactors from the connection object
	if (m_pConnection)
	{
		m_pConnection->RemoveConnectionReactor(m_pConnectionReactor);		
		m_pConnection->RemoveImportReactor(m_pImportReactor);		
		m_pConnection->RemoveExportReactor(m_pExportReactor);		
	}

	//delete all the member data, created in the constructor
	delete m_pImportReactor;
	delete m_pExportReactor;
	delete m_pConnectionReactor;
}


//this function supposed to be called when the export reactor tells us that the
//block reference is about to go to the Oracle database
//
void AttrPosEnabler::AddBlockReferenceToCache(
	AcDbBlockReference *pBlockRef, 
	OValue lOracleID)
{
	//open the block table record
	AcDbBlockTableRecord *pBTR=NULL;
	if(Acad::eOk != acdbOpenObject(
		(AcDbObject*&)pBTR, 
		pBlockRef->blockTableRecord(), 
		AcDb::kForRead))
	{
		assert(false);
		return;
	}

	//look if it has the attribute and add this object ID to the cache
	//for further processing
	if (pBTR->hasAttributeDefinitions())
	{
		m_map.insert(MapBlocks::value_type(lOracleID, pBlockRef->objectId()));
	}

	//close block table record
	if (pBTR)
	{
		pBTR->close();
	}
}

//this function supposed to be called when the export reactor signals that a
//number of objects has gone to the Oracle database. It writes the positions of
//the block attributes to the database table.
//
void AttrPosEnabler::ExportAttributePositions(std::vector<OValue> &vOracleIDs)
{
	bool bFirst=true;

	//loop through the vector of cached object IDs
	for (int i=0; i<vOracleIDs.size(); i++)
	{
		//Find out the correspondent AutoCAD ID
		AcDbObjectId id=m_map[vOracleIDs[i]];
		if (id.isNull())
		{
			continue;
		}

		//... and open the correspondent block reference
		AcDbBlockReference *bBlockRef=NULL;
		if(Acad::eOk != acdbOpenObject(
			(AcDbObject*&)bBlockRef, 
			id, 
			AcDb::kForRead,
			true))	//to be able to work with erased pbject
		{
			assert(false);
			return;
		}

		//Get the attribute iterator
		AcDbObjectIterator *pIter=bBlockRef->attributeIterator();
		assert(pIter);

		//... and loop through the attributes of the block reference
		for (pIter->start(); Adesk::kTrue!=pIter->done(); pIter->step())
		{
			//Open attribute
			AcDbAttribute *pAttrib=NULL;
			if(Acad::eOk != bBlockRef->openAttribute(
				pAttrib,
				pIter->objectId(),
				AcDb::kForRead,
				true))
			{
				delete pIter;
				bBlockRef->close();
				assert(false);
				return;
			}

			//Store the attribute positions in the cache for futher writing
			//to the database. This operation coud be done right away, but in
			//this case the it will take mach more time. Storing anything to 
			//the Oracle on row-by-row basis leads to a lignificant performance
			//degradation, so we have to do the insertion using bulk insert.
			CasheAttributePosition(pAttrib, vOracleIDs[i]);
			
			//close the attribute
			pAttrib->close();
		}

		//close the block reference and delete the attribute iterator
		bBlockRef->close();
		delete pIter;
	}

	//now we can finally save the positions of the attributes to the database
	WriteCachedPositions();
}

//This function creates the parameter array for the bulk insert operation
//
void AttrPosEnabler::CreateParameters(int nSize)
{
	assert(m_clollParams.IsOpen());
	for (int i=0;i<4; i++)
	{
		m_oParamPos[i]=m_clollParams.AddTable(
			sm_kpcParamNames[i],
			OPARAMETER_INVAR,
			sm_nParamTypes[i],
			nSize);
		assert(m_oParamPos[i].IsOpen());
	}
}

//This function creates the parameter array and write the cached attaributes' 
//positions to the database.
//
void AttrPosEnabler::WriteCachedPositions()
{
	int nSize=m_mapPos.size();
	if (!nSize)
	{
		return;
	}
	CreateParameters(nSize);
	WriteToDatabasePositions();
}

//This function is responsible for the actual writing of the information
//to the database
void AttrPosEnabler::WriteToDatabasePositions()
{
	oresult res=OFAILURE;
	int i=0;

	//loop through the cache
	for (MapPositions::const_iterator it=m_mapPos.begin();
		 it!=m_mapPos.end();
		 it++, i++)
	{
		//set the first parameter to the value of the Oracle object ID
		res=m_oParamPos[0].SetValue((it->first),i);
		assert(OSUCCESS==res);

		//and set the rest of the parameters to the values of the
		//attributes' coordinates
		for (int j=1;j<4; j++)
		{
			res=m_oParamPos[j].SetValue((double)(it->second[j-1]),i);
			assert(OSUCCESS==res);
		}
	}

	//Finally write all the array to the database in one transaction (bulk insert)
	const char *kpcSQLCreateTable=
		"insert into POSITIONS values("
		":ORAID,"
		":XPOS,"
		":YPOS,"
		":ZPOS)";
	res=m_pConnection->Database().ExecuteSQL(kpcSQLCreateTable);
	assert(OSUCCESS==res);

	//This line could be used by a developer for diagnostic purposes.
	//To make things simple we omit the error handling code, but the
	//call to the database, which allows to determine the error nature
	//is still here, so you can extend the functionality, adding some diagnostic.
	const char *err=m_pConnection->Database().GetServerErrorText();

	//This function removes the parameter arrays (in OO4O 
	//they are called parameter tables)
	RemoveTables();
}

//This function simply stores the attribute position it the cache, which contains
//the map between Oracle IDs and attribute positions\
// 
void AttrPosEnabler::CasheAttributePosition(const AcDbAttribute *pAttrib, OValue oracleId)
{
	m_mapPos.insert(MapPositions::value_type(oracleId,pAttrib->position()));
}

//This function removes the parameter arrays
//
void AttrPosEnabler::RemoveTables()
{
	for (int i=0;i<4; i++)
	{
		oresult res=m_clollParams.Remove(sm_kpcParamNames[i]);
		assert(OSUCCESS==res);
	}
}

//This function stores in the AutoCAD object id in the vector of IDs
//
void AttrPosEnabler::AddBlockIDToCache(AcDbObjectId id)
{
	m_vBlockID.push_back(id);
}

//This function is intended to be called after the import process to correct
//attribute positions (since they could not be the same as the attribute positions
//defined in the block definition.
//
void AttrPosEnabler::CorrectAttributePositions()
{
	//loop through the vector of the block IDs (Autocad object IDs)
	for (VBlocksIds::iterator it=m_vBlockID.begin(); it!=m_vBlockID.end(); it++)
	{
		//open the object
		AcDbObject* pObj=NULL;
		if(Acad::eOk != acdbOpenObject(
			pObj, 
			*it, 
			AcDb::kForRead))
		{
			assert(false);
			return;
		}

		//cast the object to the block reference
		AcDbBlockReference *pBlockRef=AcDbBlockReference::cast(pObj);
		if (!pBlockRef)
		{
			assert(false);
			return;
		}

		//find out if the block has the attributes and cache it
		if (IsHasAttributes(pBlockRef))
		{
			CacheOracleId(*it);
		}

		//finally close the object
		pObj->close();
	}

	//now go through the cached vector of the IDs and update the positions
	//of the attributes
	AttributeUpdate();
}

//This function reads the coordinates of the attributes, stored 
//in the database and update the correspondent positions of the
//attributes in the drawing.
//
void AttrPosEnabler::AttributeUpdate()
{
	//this select statement read the row from the table, which containes
	//the stored coordinates for the attributes.
	const char *kpcQuery="select * from POSITIONS where ENTITYID=%lu";

	ODynaset ds;
	//go through the cached oracle IDs
	for (int i=0; i<m_CorrectCash.size(); i++)
	{
		//execute the select statement
		char sql[100];
		sprintf(sql, kpcQuery, (long)m_CorrectCash[i].first);
		if (OSUCCESS!=ds.Open(
			m_pConnection->Database(),
			sql,
			ODYNASET_NOCACHE|ODYNASET_READONLY|ODYNASET_NOBIND))
		{
			assert(false);
			return;
		}
		
		//and correct coordinates of the attributes
		UpdateParameters(ds, m_CorrectCash[i].second);

		ds.Close();
	}
}

//This function simply reads the values (coordinates) from the dynaset
//and calls the function, which actually updates the positions in the
//drawing
// 
void AttrPosEnabler::UpdateParameters(ODynaset &ds, AcDbObjectId id)
{
	std::vector<AcGePoint3d> vPos;
	int i=0;
	for (ds.MoveFirst(); !ds.IsEOF(); ds.MoveNext(), i++)
	{
		AcGePoint3d pnt;
		ds.GetFieldValue(1, &pnt[0]);
		ds.GetFieldValue(2, &pnt[1]);
		ds.GetFieldValue(3, &pnt[2]);
		vPos.push_back(pnt);
	}
	UptatePosition(id, vPos);
}

//This function takes block reference ID, coordinates and put the attributes
//of this block to the proper positions
//
void AttrPosEnabler::UptatePosition(
	AcDbObjectId id, 
	const std::vector<AcGePoint3d> &vPos)
{
	//open block reference
	AcDbBlockReference *bBlockRef=NULL;
	if(Acad::eOk != acdbOpenObject(
		(AcDbObject*&)bBlockRef, 
		id, 
		AcDb::kForRead,
		true))	//to be able to work with erased pbject
	{
		assert(false);
		return;
	}

	//get the attribute iterator
	AcDbObjectIterator *pIter=bBlockRef->attributeIterator();
	assert(pIter);

	//go through the attributes 
	int i=0;
	for (pIter->start(); Adesk::kTrue!=pIter->done(); pIter->step(), i++)
	{
		//open attribute
		AcDbAttribute *pAttrib=NULL;
		if(Acad::eOk != bBlockRef->openAttribute(
			pAttrib,
			pIter->objectId(),
			AcDb::kForWrite,
			true))
		{
			delete pIter;
			assert(false);
			return;
		}

		//finally set the position of the attribute to the new location
		pAttrib->setPosition(vPos[i]);

		//close the attribute
		pAttrib->close();
	}
	//close the block reference
	bBlockRef->close();

	//delete the attribute iterator
	delete pIter;
}

//This function simply put in the cache (standard map) a pair of 
//the AutoCAD's object ID and Oracle object's ID
//
void AttrPosEnabler::CacheOracleId(AcDbObjectId id)
{
	AcMapOSEObject identification(*m_pConnection);
	if (identification.Init(id))
	{
		std::pair<OValue, AcDbObjectId> pair(identification.GetOracleID(),id);
		m_CorrectCash.push_back(pair);
	}
}

//This function determines whether the block reference has the attribute
//
bool AttrPosEnabler::IsHasAttributes(AcDbBlockReference *pBlockRef)
{	

	bool bRet=false;
	AcDbBlockTableRecord *pBTR=NULL;
	if(Acad::eOk != acdbOpenObject(
		(AcDbObject*&)pBTR, 
		pBlockRef->blockTableRecord(), 
		AcDb::kForRead))
	{
		assert(false);
		return false;
	}

	if (pBTR->hasAttributeDefinitions())
	{
		bRet=true;
	}

	if (pBTR)
	{
		pBTR->close();
	}
	return bRet;
}
///////////////////////////////////////////////////////////////////////////////
//								EXPORT REACTOR								 //
///////////////////////////////////////////////////////////////////////////////

//Constructor
//
MyExportReactor::MyExportReactor(AttrPosEnabler *pEnabler)
{
	m_pEnabler=pEnabler;
}

//Destructor
//
MyExportReactor::~MyExportReactor()
{
}

//This function is called by Autodesk MAP when an object goes to
//the Oracle parameter array before being exported
//
void MyExportReactor::ObjectCached(AcDbEntity *pObj, OValue lOracleID)
{
	AcDbBlockReference *pBlockRef=AcDbBlockReference::cast(pObj);

	//if it is a block reference we will store its Oracle ID for further processing
	if (pBlockRef)
	{
		m_pEnabler->AddBlockReferenceToCache(pBlockRef,lOracleID);
	}
}

//This function is called by Autodesk MAP when a number of objects have
//written to the database as a result of bulk insert.
//
void MyExportReactor::ObjectsExported(std::vector<OValue> &vOracleIDs)
{
	m_pEnabler->ExportAttributePositions(vOracleIDs);
}

///////////////////////////////////////////////////////////////////////////////
//								IMPORT REACTOR								 //
///////////////////////////////////////////////////////////////////////////////

//Constructor
//
MyImportReactor::MyImportReactor(AttrPosEnabler *pEnabler)
{
	m_pEnabler=pEnabler;
}

//Destructor
//
MyImportReactor::~MyImportReactor()
{
}

//This function is called by Autodesk MAP after a record is red from the database
//and is written to the drawing.
//
void MyImportReactor::RecordImported(const ODynaset &Dynaset, AcDbEntity *pAcDbEntity)
{
	//we have to determine whether it is a block reference and if this block
	//reference has attributes
	AcDbBlockReference *pBlockRef=AcDbBlockReference::cast(pAcDbEntity);
	if (pBlockRef)
	{
		//open block reference
		AcDbBlockTableRecord *pBTR=NULL;
		if(Acad::eOk != acdbOpenObject(
			(AcDbObject*&)pBTR, 
			pBlockRef->blockTableRecord(), 
			AcDb::kForRead))
		{
			assert(false);
			return;
		}

		//find out if it has attributes
		if (pBTR->hasAttributeDefinitions())
		{
			m_pEnabler->AddBlockIDToCache(pAcDbEntity->objectId());
		}

		//close the block reference
		pBTR->close();
	}

}

//This function is here because you might be interested in handling
//the error situation when the record is rejected (for whatever reasons)
//by Autodesk MAP import procedure. For simplicity, the error handling
//is omitted in this example.
//
void MyImportReactor::RecordRejected(const ODynaset &Dynaset)
{
}

///////////////////////////////////////////////////////////////////////////////
//								CONNECTION									 //
///////////////////////////////////////////////////////////////////////////////

//The following functions do nothing, but a user could be interested in
//handling a situation, when Autodesk MAP or other application breaks
//the connection to the Oracle database. Since the connection is shared
//between Autocad MAP and other applications it might be necessary to
//resolve the situation of loosing the connection. Application should
//never rely on the connection permanency. For simplicity, the code,
//which treates this situation is omitted, but you can extend the 
//functionality using the code below.
//
MyConnectionReactor::MyConnectionReactor(AttrPosEnabler *pEnabler)
{
}

MyConnectionReactor::~MyConnectionReactor()
{
}

void MyConnectionReactor::Disconnected()
{
}
